﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.UserModels.Filters;
    using Utilities;
    using Hims.Shared.Library.Enums;
    using Hims.Api.Models;

    /// <inheritdoc />
    [AllowAnonymous]
    [Route("api/lab-order-value")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class LabOrderValueController : BaseController
    {
        /// <summary>
        /// The ilabOrderValueService services.
        /// </summary>
        private readonly ILabOrderValueService labOrderValueService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public LabOrderValueController(ILabOrderValueService labOrderValueService, IAuditLogService auditLogServices) { this.labOrderValueService = labOrderValueService; this.auditLogServices = auditLogServices; }

        /// <summary>
        /// The fetch LabOrderValue.
        /// </summary>
        /// <param name="model">
        /// The LabOrderValue filter model.
        /// </param>
        /// <returns>
        /// The list of LabOrderValues.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of LabOrderValues.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<LabOrderValueModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody]LabOrderValueFilterModel model)
        {
            model = (LabOrderValueFilterModel)EmptyFilter.Handler(model);
            var labOrderValues = await this.labOrderValueService.FetchAsync(model);
            return this.Success(labOrderValues);
        }

        /// <summary>
        /// The add generalAdvice.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - generalAdvice added successfully.
        /// - 409 - generalAdvice already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] LabOrderValueModel model,[FromHeader]LocationHeader header)
        {
            model = (LabOrderValueModel)EmptyFilter.Handler(model);
            var response = await this.labOrderValueService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given LabOrderValue name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.LabOrderValues,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{model.Name} LabOrderValue has been added.",
                LocationId= Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("LabOrderValue has been added successfully.");
        }

        /// <summary>
        /// The update generalAdvice.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - generalAdvice updated successfully.
        /// - 409 - generalAdvice already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] LabOrderValueModel model, [FromHeader] LocationHeader header)
        {
            model = (LabOrderValueModel)EmptyFilter.Handler(model);
            var response = await this.labOrderValueService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given LabOrderValue name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.LabOrderValues,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{model.Name} LabOrderValue has been updated successfully.",
                LocationId= Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("LabOrderValue has been updated successfully.");
        }

        /// <summary>
        /// The delete generalAdvice.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - generalAdvice deleted successfully.
        /// - 409 - generalAdvice can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] LabOrderValueModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (LabOrderValueModel)EmptyFilter.Handler(model);
                var response = await this.labOrderValueService.DeleteAsync(model.LabOrderValueId);
                var responseProviderEncounter = await this.labOrderValueService.updateProviderEncounterAsync(model.LabOrderValueId);
                if (response == 0)
                {
                    return this.ServerError();
                }

                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.LabOrderValues,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow,
                    LogDescription = $"{model.Name} LabOrderValue has been deleted successfully.",
                    LocationId= Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
                return this.Success("LabOrderValue has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                return this.ServerError(exception.Message);
            }
        }
    }
}